// ==UserScript==
// @name         5ch URL補完 + リダイレクト除去 強化版
// @namespace    https://example.com/
// @version      1.8.1
// @description  ttp:// や s:// 等のURL補完 + 拡張子URL + 5chリダイレクト(jump.5ch.net)除去（遅延・監視対応）+ ゼロ幅・セミコロン除去 + リンクテキスト日本語化
// @match        *://*.5ch.net/*
// @grant        none
// @run-at       document-end
// ==/UserScript==

(function () {
    'use strict';

    const patterns = [
    // 完全な http(s):// のリンク
    {
        pattern: /\bhttps?:\/\/(?:[a-z0-9\-]+\.)+[a-z]{2,6}(?:\/[^\s<>"']*)?/gi,
        fix: url => url
    },

    // スキーム省略（protocol-relative）リンク: //example.com
    {
        pattern: /(?<!:)\b\/\/(?:[a-z0-9\-]+\.)+[a-z]{2,6}(?:\/[^\s<>"']*)?/gi,
        fix: url => 'https:' + url
    },

    // 不完全な https 省略: tps://example.com → https://example.com
    {
        pattern: /\btps:\/\/(?:[a-z0-9\-]+\.)+[a-z]{2,6}[^\s<>"']*/gi,
        fix: url => 'https://' + url.slice(6)
    },
    {
        pattern: /\bhtps:\/\/(?:[a-z0-9\-]+\.)+[a-z]{2,6}[^\s<>"']*/gi,
        fix: url => 'https://' + url.slice(6)
    },
    {
        pattern: /\bps:\/\/(?:[a-z0-9\-]+\.)+[a-z]{2,6}[^\s<>"']*/gi,
        fix: url => 'https://' + url.slice(5)
    },
    {
        pattern: /\bs:\/\/(?:[a-z0-9\-]+\.)+[a-z]{2,6}[^\s<>"']*/gi,
        fix: url => 'https://' + url.slice(4)
    },

    // 不完全な http 省略: ttp://example.com → http://example.com
    {
        pattern: /\bttp:\/\/(?:[a-z0-9\-]+\.)+[a-z]{2,6}[^\s<>"']*/gi,
        fix: url => 'http://' + url.slice(6)
    },
    {
        pattern: /\btp:\/\/(?:[a-z0-9\-]+\.)+[a-z]{2,6}[^\s<>"']*/gi,
        fix: url => 'http://' + url.slice(5)
    },
    {
        pattern: /\bp:\/\/(?:[a-z0-9\-]+\.)+[a-z]{2,6}[^\s<>"']*/gi,
        fix: url => 'http://' + url.slice(4)
    },

    // www.から始まる省略形
    {
        pattern: /(?<!https?:)\/\/?(www\.[a-z0-9\-]+\.[a-z]{2,6}[^\s<>"']*)/gi,
        fix: url => 'https://' + url.replace(/^\/\//, '')
    },

    // ドメイン + 拡張子付きファイル（画像/動画など）
    {
        pattern: /\b([a-z0-9\-]+\.)+[a-z]{2,6}\/[^\s<>"']+\.(jpg|jpeg|png|gif|webp|bmp|svg|mp4|mov|avi)(\?[^\s<>"']*)?/gi,
        fix: url => 'https://' + url
    },

    // 最後に：ドメイン + パス（上記すべてに該当しないがドメイン構造を持つもの）
    {
        pattern: /\b([a-z0-9\-]+\.)+[a-z]{2,6}\/[^\s<>"']+/gi,
        fix: url => 'https://' + url
    }
];

    function isInsideLink(node) {
        while (node) {
            if (node.nodeType === 1 && node.tagName === 'A') return true;
            node = node.parentNode;
        }
        return false;
    }

    function linkifyTextNode(node) {
        if (!node.nodeValue) return;
        const parent = node.parentNode;
        if (!parent || /^(SCRIPT|STYLE|TEXTAREA|INPUT)$/i.test(parent.tagName)) return;
        if (isInsideLink(node)) return;

        let text = node.nodeValue;
        const fragments = [];

        while (text.length > 0) {
            let earliestMatch = null;
            for (const { pattern, fix } of patterns) {
                pattern.lastIndex = 0;
                const m = pattern.exec(text);
                if (m) {
                    const idx = m.index;
                    if (earliestMatch === null || idx < earliestMatch.index) {
                        earliestMatch = {
                            index: idx,
                            length: m[0].length,
                            match: m[0],
                            fix: fix
                        };
                    }
                }
            }

            if (!earliestMatch) {
                fragments.push(document.createTextNode(text));
                break;
            }

            const { index, length, match, fix } = earliestMatch;

            if (index > 0) {
                fragments.push(document.createTextNode(text.slice(0, index)));
            }

            // 末尾にある不要文字を除去（ゼロ幅スペースなども）
            const trailingPunct = /[。、．。！？,.!?！？;；\u200B\uFEFF\u202C]+$/u;
            const trimmedMatch = match.replace(trailingPunct, '');
            const href = fix(trimmedMatch);

            const a = document.createElement('a');
            a.href = href;
            a.target = '_blank';
            a.rel = 'noopener noreferrer';
            a.textContent = trimmedMatch;
            fragments.push(a);

            const trailing = match.slice(trimmedMatch.length);
            if (trailing) {
                fragments.push(document.createTextNode(trailing));
            }

            text = text.slice(index + length);
        }

        if (fragments.length > 0) {
            fragments.forEach(fragment => parent.insertBefore(fragment, node));
            parent.removeChild(node);
        }
    }

    function linkifyTextNodes(root) {
        const walker = document.createTreeWalker(root, NodeFilter.SHOW_TEXT, null);
        const textNodes = [];
        while (walker.nextNode()) {
            textNodes.push(walker.currentNode);
        }
        for (const node of textNodes) {
            linkifyTextNode(node);
        }
    }

    // 🛡 5chリダイレクト除去処理
    function remove5chRedirects(root) {
        const anchors = root.querySelectorAll('a[href*="jump.5ch.net/?"]');
        anchors.forEach(a => {
            try {
                const url = new URL(a.href);
                const actual = decodeURIComponent(url.search.slice(1));
                if (actual.startsWith('http')) {
                    a.href = actual;
                    a.rel = 'noopener noreferrer';
                }
            } catch (e) {
                console.warn('Jump URL処理失敗:', a.href);
            }
        });
    }

    // HTMLデコード関数を追加
    function decodeHTML(str) {
        const txt = document.createElement('textarea');
        txt.innerHTML = str;
        return txt.value;
    }

    // 🔍 hrefおよびテキストから不可視文字やセミコロンを除去（修正版）
    function cleanAnchorUrls(root) {
        const anchors = root.querySelectorAll('a[href]');
        anchors.forEach(a => {
            try {
                let rawHref = a.getAttribute('href');
                if (!rawHref) return;

                // デコードしてから不要文字除去
                const decodedHref = decodeHTML(rawHref);
                const cleanedHref = decodedHref.replace(/[\u200B\uFEFF\u202C;]/g, '');

                if (decodedHref !== cleanedHref) {
                    a.href = cleanedHref;
                    a.setAttribute('href', cleanedHref);
                }

                // textContentはエンティティデコード済みなのでそのまま置換
                const cleanedText = a.textContent.replace(/[\u200B\uFEFF\u202C]/g, '').replace(/[。、．。！？,.!?！？;；]+$/u, '');
                if (a.textContent !== cleanedText) {
                    a.textContent = cleanedText;
                }

                const next = a.nextSibling;
                if (next && next.nodeType === Node.TEXT_NODE) {
                    next.textContent = next.textContent.replace(/^[;；\s\u200B\uFEFF\u202C]+/, '');
                }
            } catch (e) {
                console.warn('cleanAnchorUrls失敗:', a.href, e);
            }
        });
    }

    // 追加: リンクテキストを日本語化しつつhrefはエンコードされたURLに修正
    function fixAnchorTextAndHref(root) {
        const anchors = root.querySelectorAll('a[href]');
        const excludeRegex = /^>>?\d+[\p{L}\p{N}_\-]*$/u;

        anchors.forEach(a => {
            try {
                const originalText = a.textContent.trim();

                // >>数字やそのパターンはスキップ
                if (excludeRegex.test(originalText)) return;

                const url = new URL(a.href);
                const encodedHref = url.href;

                // 元テキストからスキーム部分を抽出（例: "ttps://", "https://" など）
                const schemeMatch = originalText.match(/^[a-z]+:\/\//i);
                const scheme = schemeMatch ? schemeMatch[0] : '';

                // スキーム以降のテキスト部分
                const restText = scheme ? originalText.slice(scheme.length) : originalText;

                // restText を decodeURIComponent でデコード（失敗したら元のまま）
                let decodedRest;
                try {
                    decodedRest = decodeURIComponent(restText);
                } catch {
                    decodedRest = restText;
                }

                // 表示テキストをスキーム + 日本語化済みの部分で作成
                const displayText = scheme + decodedRest;

                // hrefは正規化済みのURLをセット
                if (a.href !== encodedHref) {
                    a.href = encodedHref;
                }

                // テキストを更新（元と違うなら）
                if (a.textContent !== displayText) {
                    a.textContent = displayText;
                }
            } catch (e) {
                // URL不正などのエラーは無視
            }
        });
    }

    function processPostContent(root) {
        linkifyTextNodes(root);
        remove5chRedirects(root);
        cleanAnchorUrls(root);
        fixAnchorTextAndHref(root);
    }

    // 初期読み込み後
    const initialPosts = document.querySelectorAll('.post-content');
    initialPosts.forEach(post => processPostContent(post));

    // 遅延読み込み対応（強制再処理）
    setTimeout(() => {
        document.querySelectorAll('.post-content').forEach(post => processPostContent(post));
    }, 1500);

    // 動的追加の監視
    const observer = new MutationObserver(mutations => {
        for (const mutation of mutations) {
            for (const node of mutation.addedNodes) {
                if (node.nodeType !== 1) continue;
                if (node.classList.contains('post-content')) {
                    processPostContent(node);
                } else {
                    node.querySelectorAll?.('.post-content').forEach(child => processPostContent(child));
                }
            }
        }
    });

    observer.observe(document.body, {
        childList: true,
        subtree: true
    });
})();
